// Copyright ® 2025 OneSpan North America, Inc. All rights reserved. 

 
/////////////////////////////////////////////////////////////////////////////
//
//
// This file is example source code. It is provided for your information and
// assistance. See your licence agreement for details and the terms and
// conditions of the licence which governs the use of the source code. By using
// such source code you will be accepting these terms and conditions. If you do
// not wish to accept these terms and conditions, DO NOT OPEN THE FILE OR USE
// THE SOURCE CODE.
//
// Note that there is NO WARRANTY.
//
//////////////////////////////////////////////////////////////////////////////


import AVFoundation

protocol CameraManagerDelegate: AnyObject {
    func cameraManager(didOutput sampleBuffer: CMSampleBuffer)
    func cameraManager(didFailWithError error: CameraManagerError)
}

class CameraManager: NSObject, ObservableObject {
    private enum Status {
        case unconfigured
        case configured
        case unauthorized
        case failed
    }
    
    weak var delegate: CameraManagerDelegate?
    let session = AVCaptureSession()
    
    private let videoOutput = AVCaptureVideoDataOutput()
    private var status: Status = .unconfigured
    private let sessionQueue = DispatchQueue(label: "combine_scanner")
    private let videoOutputQueue = DispatchQueue(label: "video_output",
                                                 qos: .userInitiated,
                                                 attributes: [],
                                                 autoreleaseFrequency: .workItem)
    override init() {
        super.init()
                
        set(self, queue: videoOutputQueue)
        config()
    }
    
    private func config() {
        checkPermissions()
        
        guard status != .unauthorized else { return }
        sessionQueue.async {
            self.configCaptureSession()
            self.startSession()
        }
    }
    
    private func configCaptureSession() {
        guard status == .unconfigured else {
            return
        }
        session.beginConfiguration()
        defer {
            session.commitConfiguration()
        }
        
        session.sessionPreset = .hd1280x720
        
        //Preparing The device as input
        let device = AVCaptureDevice.default(for: .video)
        guard let camera = device else {
            delegate?.cameraManager(didFailWithError: .cameraUnavailable)
            status = .failed
            return
        }
        
        //Adding input device to session
        do {
            let cameraInput = try AVCaptureDeviceInput(device: camera)
            if session.canAddInput(cameraInput) {
                session.addInput(cameraInput)
            } else {
                delegate?.cameraManager(didFailWithError: .cannotAddInput)
                status = .failed
                return
            }
        } catch let err {
            delegate?.cameraManager(didFailWithError: .cameraInputFailure(err))
            status = .failed
            return
        }
        
        //Adding output to session
        if session.canAddOutput(videoOutput) {
            session.addOutput(videoOutput)
            
            //set the format type for the video output.
            videoOutput.videoSettings = [kCVPixelBufferPixelFormatTypeKey as String: kCVPixelFormatType_32BGRA]
            
            // force the orientation to be in portrait. to create vertical videos
            let videoConnection = videoOutput.connection(with: .video)
            videoConnection?.videoOrientation = .portrait
        } else {
            delegate?.cameraManager(didFailWithError: .cannotAddOutput)
            status = .failed
            return
        }
        
        status = .configured
    }
    
    private func controlSession(start: Bool) {
        guard status == .configured else {
            config()
            return
        }
        
        sessionQueue.async {
            if start {
                if !self.session.isRunning {
                    self.session.startRunning()
                }
            } else {
                self.session.stopRunning()
            }
        }
    }
    
    // MARK: - Public methods
    func stopSession() {
        controlSession(start: false)
    }
    
    func startSession() {
        controlSession(start: true)
    }
}

// MARK: - Permissions
extension CameraManager {
    private func checkPermissions() {
        switch AVCaptureDevice.authorizationStatus(for: .video) {
        case .notDetermined:

        sessionQueue.suspend()
        AVCaptureDevice.requestAccess(for: .video) { authorized in
 
            if !authorized {
                self.status = .unauthorized
                self.delegate?.cameraManager(didFailWithError: .deniedAuthorization)
            }
            self.sessionQueue.resume()
        }

        case .restricted:
            status = .unauthorized
            delegate?.cameraManager(didFailWithError: .restrictedAuthorization)
        case .denied:
            status = .unauthorized
            delegate?.cameraManager(didFailWithError: .deniedAuthorization)

        case .authorized:
            break

        @unknown default:
            status = .unauthorized
            delegate?.cameraManager(didFailWithError: .unknownAuthorization)
        }
    }
}

// MARK: - Video buffer output
extension CameraManager: AVCaptureVideoDataOutputSampleBufferDelegate {
    func captureOutput(
        _ output: AVCaptureOutput,
        didOutput sampleBuffer: CMSampleBuffer,
        from connection: AVCaptureConnection
    ){
        delegate?.cameraManager(didOutput: sampleBuffer)
    }
    
    func set(
        _ delegate: AVCaptureVideoDataOutputSampleBufferDelegate,
        queue: DispatchQueue
    ){
        sessionQueue.async {
            self.videoOutput.setSampleBufferDelegate(delegate, queue: queue)
        }
    }
}
